#!/usr/bin/env python3
import sys, json, math

def approx_leq(a, b, tol=1e-9):
    return a <= b + tol

def main(results_path, requirements_path):
    with open(results_path, 'r', encoding='utf-8') as f:
        R = json.load(f)
    with open(requirements_path, 'r', encoding='utf-8') as f:
        req = json.load(f)

    ok = True

    # --- Touch
    t = R.get("touch", {})
    rq = req["touch"]
    if rq["s_emp_equals_2sigma"]:
        # Expect s_emp map with sigma->value and a sigma_values list
        sigmas = t.get("sigma_values", [])
        s_emp = t.get("s_emp", {})
        for s in sigmas:
            exp = 2*int(s)
            got = int(s_emp.get(str(s), -999))
            if got != exp:
                print(f"[TOUCH] sigma={s}: expected {exp}, got {got}")
                ok = False
    if rq["recenter_ok"] and not t.get("recenter_ok", False):
        print("[TOUCH] recenter_ok=false")
        ok = False

    # --- Vision
    v = R.get("vision", {})
    rv = req["vision"]
    # Check relative error per Theta
    if "f_pred" in v and "f_c_emp" in v and rv["relative_error_max"] is not None:
        maxerr = rv["relative_error_max"]
        for k, pred in v["f_pred"].items():
            emp = v["f_c_emp"].get(k, None)
            if emp is None:
                print(f"[VISION] missing f_c_emp for Theta={k}")
                ok = False
                continue
            err = abs(emp - pred)/pred if pred != 0 else 1e9
            if err > maxerr:
                print(f"[VISION] Theta={k}: rel error {err:.3f} > {maxerr}")
                ok = False
    if rv["recenter_ok"] and not v.get("recenter_ok", False):
        print("[VISION] recenter_ok=false")
        ok = False

    # --- Sound
    s = R.get("sound", {})
    rs = req["sound"]
    m = s.get("loglog_slope", None)
    if m is None or not (rs["slope_range"][0] <= m <= rs["slope_range"][1]):
        print(f"[SOUND] slope {m} not in {tuple(rs['slope_range'])}")
        ok = False
    if rs["recenter_ok"] and not s.get("recenter_ok", False):
        print("[SOUND] recenter_ok=false")
        ok = False
    if "preflight" in s and "noise_only_detect_rate" in s["preflight"]:
        p = s["preflight"]["noise_only_detect_rate"]
        if abs(p - rs["pFA"]) > 0.005:
            print(f"[SOUND] noise-only pFA {p:.3f} != target {rs['pFA']:.3f} (±0.005)")
            ok = False

    if ok:
        print("[OK] All acceptance checks passed.")
        sys.exit(0)
    else:
        sys.exit(1)

if __name__ == "__main__":
    if len(sys.argv) != 3:
        print("Usage: check_acceptance.py <results.json> <requirements.json>")
        sys.exit(2)
    main(sys.argv[1], sys.argv[2])
